
import ij.*;
import ij.process.*;
import ij.gui.*;
import java.awt.*;
import ij.plugin.filter.*;
import com.ducret.resultJ.*;
import java.util.Arrays;

//Plugin to equalize stack of image
public class ChannelAligner_ implements ExtendedPlugInFilter, DialogListener {

    private ImagePlus imp;
    private ImageStack stack;
    private static final String[] POSITIONS = new String[]{"auto", "slices", "frames"};
    private static final String[] CHANNELS = new String[]{"current", "all"};
    private static final String[] METHODS = ImageProcessor.getInterpolationMethods();

    private static double xOffset;
    private static double yOffset;

    private static int nbChannels;
    private static int nbSlices;
    private static int nbFrames;

    private static int refChannel;
    private static int refSlice;
    private static int refFrame;

    private static int method = 0;
    private static int position = 0;

    private static boolean applyToAll = false;
    private static boolean applyToActive = false;
    //private static int channel = 0;

    private ImageProcessor[] cProcessor;
    private boolean[] cActive;
    private String[] cLabel;
    private boolean[] cVisible;

    private LUT[] luts;
    private Color cColor;
    private int compositeMode;

    private GenericDialog gd;
    private PlugInFilterRunner pfr;
    private int flags = DOES_8G + DOES_16 + DOES_32 + DOES_STACKS;

    private boolean done;

    @Override
    public int setup(String arg, ImagePlus imp) {

        if (imp == null) {
            return flags;
        }

        this.imp = imp;

        nbChannels = imp.getNChannels();
        nbSlices = imp.getNSlices();
        nbFrames = imp.getNFrames();

        refChannel = imp.getC();
        refSlice = imp.getZ();
        refFrame = imp.getT();

        cProcessor = new ImageProcessor[nbChannels];
        cActive = new boolean[nbChannels];
        cLabel = new String[nbChannels];
        cVisible = new boolean[nbChannels];

        stack = imp.getImageStack();
        int index;
        for (int i = 0; i < nbChannels; i++) {
            index = imp.getStackIndex((i + 1), refSlice, refFrame);
            cProcessor[i] = stack.getProcessor(index);
            cLabel[i] = "ch. " + (i + 1);
            cActive[i] = ((i + 1) == refChannel);
            // ImProcessor.show("c" + (i + 1), temp[i]);
        }

        luts = ImPlus.getLuts(imp);

        if (imp.isComposite()) {
            CompositeImage imp2 = (CompositeImage) imp;
            boolean[] cTemp = imp2.getActiveChannels();
            cVisible = Arrays.copyOf(cTemp, cTemp.length);

            compositeMode = imp2.getMode();
            imp2.setMode(CompositeImage.COMPOSITE);
            if (compositeMode == 1) {
                setActiveChannels(imp2, cVisible);
            }
            imp2.setLuts(luts);
            imp2.updateAndDraw();
        }

        done = false;
        return flags;
    }

    public void resetCompositeMode() {
        resetCompositeMode(imp, compositeMode, luts, cVisible);
    }

    public static void resetCompositeMode(ImagePlus imp, int compositeMode, LUT[] luts, boolean[] cVisible) {
        if (imp.isComposite()) {
            CompositeImage imp2 = (CompositeImage) imp;
            imp2.setMode(compositeMode);
            imp2.setLuts(luts);
            setActiveChannels(imp2, cVisible);
            imp2.updateAndDraw();
        }
    }

    public static void setActiveChannels(CompositeImage imp, boolean[] cVisible) {
        boolean[] cTemp = imp.getActiveChannels();
        for (int i = 0; i < cVisible.length; i++) {
            if (i < cTemp.length) {
                cTemp[i] = cVisible[i];
            }
        }
    }

    @Override
    public void run(ImageProcessor ip) {
        if (gd.wasOKed()) {
            if (!done) {
                //reset translation of the active img
                translate(0, 0);
                if (applyToAll) {
                    ImagePlus[] imgs = ImPlus.getOpenedImagePlus();
                    for (ImagePlus i : imgs) {
                        LUT[] aLuts = ImPlus.getLuts(i);
                        int cMode;
                        boolean[] cVisible;
                        if (i.isComposite()) {
                            cMode = ((CompositeImage) i).getMode();
                            cVisible = ((CompositeImage) i).getActiveChannels();
                        } else {
                            cMode = 0;
                            cVisible = new boolean[0];
                        }
                        translate(i, xOffset, yOffset, cActive, method, position, applyToActive);
                        resetCompositeMode(i, cMode, aLuts, cVisible);
                    }
                } else {
                    translate(imp, xOffset, yOffset, cActive, method, position, applyToActive);
                    resetCompositeMode();
                }
            }
            done = true;
        } else if (gd.wasCanceled()) {
            translate(0, 0);
            resetCompositeMode();
        } else {
            translate(xOffset, yOffset);
        }
    }

    private void translate(double xOffset, double yOffset) {
        int index;
        for (int i = 0; i < nbChannels; i++) {
            index = imp.getStackIndex((i + 1), refSlice, refFrame);
            ImageProcessor ip2 = cProcessor[i].duplicate();
            if (cActive[i]) {
                ip2.translate(xOffset, yOffset);
            }
            stack.setProcessor(ip2, index);
        }
        imp.setStack(stack);
        imp.updateAndDraw();
    }

    @Override
    public int showDialog(ImagePlus imp, String command, PlugInFilterRunner pfr) {
        this.pfr = pfr;

        method = (int) Prefs.get("MicrobeJ.translate.method", 0);
        position = (int) Prefs.get("MicrobeJ.translate.position", 0);
        // channel = (int) Prefs.get("MicrobeJ.translate.channel", 0);
        xOffset = (int) Prefs.get("MicrobeJ.translate.xOffset", 0);
        yOffset = (int) Prefs.get("MicrobeJ.translate.yOffset", 0);

        applyToAll = (boolean) Prefs.get("MicrobeJ.translate.applyToAll", false);
        applyToActive = (boolean) Prefs.get("MicrobeJ.translate.applyToActive", false);
        gd = new GenericDialog("Channel Aligner");

        if (nbChannels > 1) {
            String[] cHeadings = new String[]{"Channels"};
            gd.setInsets(0, 20, 0);
            gd.addCheckboxGroup(nbChannels, 1, cLabel, cActive, cHeadings);
        }

        int digits = xOffset == (int) xOffset && yOffset == (int) yOffset ? 1 : 3;
        gd.addNumericField("X offset (pixels): ", xOffset, digits, 8, "");
        gd.addNumericField("Y offset (pixels): ", yOffset, digits, 8, "");
        gd.addChoice("Method: ", METHODS, METHODS[method]);

        if (nbSlices > 1 && nbFrames > 1) {
            gd.addChoice("Positions: ", POSITIONS, POSITIONS[position]);
        }
        gd.addCheckbox("Apply to Opened Img.", applyToAll);
        gd.addCheckbox("Apply to Active Pos.", applyToActive);

        gd.addPreviewCheckbox(pfr, "Preview");
        gd.addDialogListener(this);

        gd.showDialog();

        if (gd.wasCanceled()) {
            resetCompositeMode();
            return DONE;
        }
        return IJ.setupDialog(imp, flags);
    }

    @Override
    public boolean dialogItemChanged(GenericDialog gd, AWTEvent e) {

        if (nbChannels > 1) {
            for (int i = 0; i < nbChannels; i++) {
                cActive[i] = gd.getNextBoolean();
            }
        }

        xOffset = gd.getNextNumber();
        yOffset = gd.getNextNumber();
        method = gd.getNextChoiceIndex();

        if (nbSlices > 1 && nbFrames > 1) {
            position = gd.getNextChoiceIndex();
        } else {
            position = 0;
        }

        applyToAll = gd.getNextBoolean();
        applyToActive = gd.getNextBoolean();

        Prefs.set("MicrobeJ.translate.method", method);
        Prefs.set("MicrobeJ.translate.position", position);
        //Prefs.set("MicrobeJ.equalize.channel", channel);
        Prefs.set("MicrobeJ.translate.xOffset", xOffset);
        Prefs.set("MicrobeJ.translate.yOffset", yOffset);
        Prefs.set("MicrobeJ.translate.applyToAll", applyToAll);
        Prefs.set("MicrobeJ.translate.applyToActive", applyToActive);

        if (gd.invalidNumber()) {
            if (gd.wasOKed()) {
                IJ.error("Offset is invalid.");
            }
            return false;
        }

        return true;
    }

    public void setNPasses(int nPasses) {
    }

    public static void translate(ImagePlus imp, double xOffset, double yOffset, boolean[] channels, int method, int positionMode, boolean activePosition) {

        ImageStack stack = imp.getStack();
        int nbSlices = imp.getNSlices();
        int nbFrames = imp.getNFrames();
        int nbChannels = imp.getNChannels();

        //RJ.l(">s:" + nbSlices + " / f:" + nbFrames + " / c:" + nbChannels);
        channels = Arrays.copyOf(channels, nbChannels);

        int position = 0;
        int pMode;
        switch (positionMode) {
            case 1://slices
                position = nbSlices;
                pMode = 1;
                break;
            case 2://frames
                position = nbFrames;
                pMode = 2;
                break;
            default:
                if (nbSlices == 1 && nbFrames > 1) {
                    pMode = 2;
                } else if (nbFrames == 1 && nbSlices > 1) {
                    pMode = 1;
                } else {
                    pMode = 0;
                }
                position = nbFrames * nbSlices;
                break;
        }

        int index;
        ImageProcessor ip;
        Roi r = imp.getRoi();
        imp.setRoi((Roi) null);
        IJ.showProgress(0);
        int slice;
        int frame;
        if (activePosition) {
            slice = imp.getSlice();
            frame = imp.getFrame();
            for (int c = 1; c <= nbChannels; c++) {
                if (channels[c - 1]) {
                    index = imp.getStackIndex(c, slice, frame);
                    ip = stack.getProcessor(index);
                    ip.setInterpolationMethod(method);
                    ip.translate(xOffset, yOffset);
                    stack.setProcessor(ip, index);
                    IJ.showProgress((double) c / nbChannels);
                }
            }
        } else {
            slice = 1;
            frame = 1;
            for (int c = 1; c <= nbChannels; c++) {
                if (channels[c - 1]) {
                    for (int i = 1; i <= position; i++) {
                        index = imp.getStackIndex(c, slice, frame);
                        //RJ.l(index + ">" + c + " : " + slice + " : " + frame);
                        ip = stack.getProcessor(index);
                        ip.setInterpolationMethod(method);
                        ip.translate(xOffset, yOffset);

                        stack.setProcessor(ip, index);

                        switch (pMode) {
                            case 1://slices
                                slice++;
                                if (slice > nbSlices) {
                                    slice = 1;
                                }
                                break;
                            case 2://frames
                                frame++;
                                if (frame > nbFrames) {
                                    frame = 1;
                                }
                                break;
                            default:
                                slice++;
                                if (slice > nbSlices) {
                                    slice = 1;
                                    frame++;
                                }
                                break;
                        }

                        //slice += ((frameStack) ? 0 : 1);
                        //frame += ((frameStack) ? 1 : 0);
                        IJ.showProgress((double) i / position);
                    }
                }
            }
        }
        imp.setStack(stack);
        imp.setRoi(r);
        imp.updateAndDraw();
    }

}
