
import com.ducret.resultJ.DoublePolygon;
import static com.ducret.resultJ.DoublePolygon.POLYGON;
import com.ducret.resultJ.RJ;
import ij.*;
import ij.process.*;
import ij.gui.*;
import ij.plugin.*;
import ij.plugin.filter.*;
import ij.measure.*;
import ij.plugin.frame.*;
import java.awt.Color;
import java.awt.image.ColorModel;
import java.awt.image.IndexColorModel;
import java.util.Arrays;

public class KymoGraph_ implements PlugInFilter {

    ImagePlus imp;
    public final static int HYPERSTACK_DEFAULT = 0;
    public final static int HYPERSTACK_SLICE = 1;
    public final static int HYPERSTACK_FRAME = 2;
    public static final int BYTE_PROCESSOR = 0;
    public static final int SHORT_PROCESSOR = 1;
    public static final int FLOAT_PROCESSOR = 2;
    public static final int COLOR_PROCESSOR = 3;
    private int lineWidth;
    private int mode;
    private int projection;
    private int smooth;
    private boolean landscape;
    private boolean roiManager;
    private boolean calibration;
    private double scaling;

    private static final String[] positions = new String[]{"both", "slices", "frames"};
    private static final String[] modes = new String[]{"default", "mean", "max", "min"};
    private static final String[] postProcessing = new String[]{"default", "none", "smooth"};

    @Override
    public int setup(String arg, ImagePlus imp) {
        this.imp = imp;
        return DOES_8G + DOES_16 + DOES_32;//+ STACK_REQUIRED;
    }

    @Override
    public void run(ImageProcessor ip) {
        Roi r = imp.getRoi();
//        if (!(r == null)) {
//            int roiType = r.getType();
//            if (!(roiType == Roi.LINE || roiType == Roi.POLYLINE)) {
//                IJ.error("Line or rectangular selection required.");
//                return;
//            }
//        } else {
//            IJ.error("Selection required");
//            return;
//        }

        int nbSlices = imp.getNSlices();
        int nbFrames = imp.getNFrames();

        lineWidth = (int) Prefs.get("MicrobeJ.kymograph.lineWidth", 1);
        mode = (int) Prefs.get("MicrobeJ.kymograph.mode", 0);
        projection = (int) Prefs.get("MicrobeJ.kymograph.projection", 0);
        smooth = (int) Prefs.get("MicrobeJ.kymograph.smooth", 0);
        landscape = (boolean) Prefs.get("MicrobeJ.kymograph.landscape", true);
        roiManager = (boolean) Prefs.get("MicrobeJ.kymograph.roiManager", false);
        calibration = (boolean) Prefs.get("MicrobeJ.kymograph.calibration", false);
        scaling = (double) Prefs.get("MicrobeJ.kymograph.scaling", 1);

        lineWidth = (r != null && r.getStrokeWidth() > 0) ? (int) r.getStrokeWidth() : lineWidth;

        GenericDialog gd = new GenericDialog("KymoGraph");
        gd.addNumericField("Line Width:", lineWidth, 0);

        gd.addChoice("Mode:", modes, modes[mode]);
        if (nbSlices > 1 && nbFrames > 1) {
            gd.addChoice("Projection:", positions, positions[projection]);
        }
        gd.addChoice("Smooth", postProcessing, postProcessing[smooth]);
        gd.addCheckbox("Landscape", landscape);
        gd.addCheckbox("RoiManager", roiManager);
        gd.addCheckbox("Calibration", calibration);
        gd.addNumericField("Scaling", scaling, 1);

        gd.showDialog();
        if (!gd.wasCanceled()) {
            lineWidth = (int) gd.getNextNumber();
            mode = gd.getNextChoiceIndex();
            projection = 0;
            if (nbSlices > 1 && nbFrames > 1) {
                projection = gd.getNextChoiceIndex();
            }
            smooth = gd.getNextChoiceIndex();
            landscape = gd.getNextBoolean();
            roiManager = gd.getNextBoolean();
            calibration = gd.getNextBoolean();
            scaling = gd.getNextNumber();

            Prefs.set("MicrobeJ.kymograph.lineWidth", lineWidth);
            Prefs.set("MicrobeJ.kymograph.mode", mode);
            Prefs.set("MicrobeJ.kymograph.projection", projection);
            Prefs.set("MicrobeJ.kymograph.smooth", smooth);
            Prefs.set("MicrobeJ.kymograph.landscape", landscape);
            Prefs.set("MicrobeJ.kymograph.roiManager", roiManager);
            Prefs.set("MicrobeJ.kymograph.calibration", calibration);
            Prefs.set("MicrobeJ.kymograph.scaling", scaling);
            Prefs.savePreferences();

            int currentChannel = imp.getC();
            int currentSlice = imp.getZ();
            int currentFrame = imp.getT();

            ImagePlus output = null;

            if (roiManager) {
                RoiManager manager = RoiManager.getInstance();
                //ArrayList<Roi> output = new ArrayList<Roi>();
                if (manager != null) {
                    output = getKymograph(imp, manager.getRoisAsArray(), lineWidth, mode, smooth != 1, landscape, projection, calibration, scaling);
                }
            } else {
                output = getKymograph(imp, new Roi[]{r}, lineWidth, mode, smooth != 1, landscape, projection, calibration, scaling);
            }

            if (output != null) {
                String t = imp.getTitle();
                int index = t.lastIndexOf(".");
                if (index != -1) {
                    output.setTitle(t.substring(0, index) + "_k" + t.substring(index));
                } else {
                    output.setTitle(t + "_k");
                }
                output.show();
            }

            imp.setPosition(currentChannel, currentSlice, currentFrame);

        }
    }

    public static ImagePlus getKymograph(ImagePlus imp, Roi[] rois, int width, int mode, boolean smooth, boolean landscape, int projection, boolean calibration, double scaling) {
        int currentSlice = imp.getZ();
        int currentFrame = imp.getT();
        LUT[] luts = imp.getLuts();
        ImagePlus[] channels = ChannelSplitter.split(imp);
        ImagePlus[] output = new ImagePlus[channels.length];
        Calibration cal = imp.getCalibration();
        double calx = cal.pixelWidth;
        //double caly = cal.pixelHeight;
        double calz = cal.pixelDepth;
        double r = calibration ? calz / calx : 1;

        //Range[] displayRange = new Range[channels.length];
        double[] rangeMin = new double[channels.length];
        double[] rangeMax = new double[channels.length];

        //IJ.log(calx + " / " + caly + " / " + calz);
        for (int c = 0; c < channels.length; c++) {
            ImageStack stack = channels[c].getImageStack();

            rangeMin[c] = channels[c].getDisplayRangeMin();
            rangeMax[c] = channels[c].getDisplayRangeMax();

            //displayRange[c] = new Range(channels[c].getDisplayRangeMin(), channels[c].getDisplayRangeMax());
            int nbPosition;
            switch (projection) {
                case 1://slices
                    nbPosition = channels[c].getNSlices();
                    break;
                case 2://frames
                    nbPosition = channels[c].getNFrames();
                    break;
                default:
                    nbPosition = stack.getSize();
                    break;
            }
            //int nbPosition = channels[c].getStackSize();
            //IJ.log(projection + " nbP>" + channels[c].getNSlices() + " / " + channels[c].getNFrames());
            double[] pLine;
            int roiLength = 0;
            ImageProcessor[] processors = new ImageProcessor[rois.length];
            int maxWidth = 0;
            int minWidth = Integer.MAX_VALUE;
            int maxHeight = 0;
            int minHeight = Integer.MAX_VALUE;
            int t;

            for (int k = 0; k < rois.length; k++) {
                double[] pLines = new double[0];
                t = (width <= 0) ? (int) rois[k].getStrokeWidth() : width;

                for (int i = 1; i <= nbPosition; i++) {
                    int index;
                    switch (projection) {
                        case 1://slices
                            index = channels[c].getStackIndex(1, i, currentFrame);
                            break;
                        case 2://frames
                            index = channels[c].getStackIndex(1, currentSlice, i);
                            break;
                        default:
                            index = i;
                            break;
                    }
                    //IJ.log(">" + index);
                    pLine = getProfile(stack.getProcessor(index), rois[k], t, mode);
                    if (i == 1) {
                        roiLength = pLine.length;
                        pLines = new double[nbPosition * roiLength];
                    }
                    for (int j = 0; j < roiLength; j++) {
                        pLines[(i - 1) * roiLength + j] = pLine[j];
                    }
                }
                //IJ.log(">" + roiLength + " : " + nbPosition);
                int w = roiLength;
                int h = nbPosition;
                FloatProcessor kIp = new FloatProcessor(w, h, pLines);
                kIp.setInterpolate(true);
                kIp.setInterpolationMethod(ImageProcessor.BILINEAR);
                int w2 = w;
                int h2 = (int) Math.round((double) ((nbPosition - 1) * r) * scaling);

                maxWidth = Math.max(maxWidth, w2);
                minWidth = Math.min(minWidth, w2);
                maxHeight = Math.max(maxHeight, h2);
                minHeight = Math.min(minHeight, h2);

                if (smooth) {
                    kIp.smooth();
                }
                processors[k] = kIp.resize(w2, h2);//ImageProcessor kIp2
                //processors[k] = kIp2.convertToShort(true);
            }

            ImageProcessor kIp3;
            if (processors.length > 1) {
                ImageStack stack2 = (landscape) ? new ImageStack(maxHeight, maxWidth) : new ImageStack(maxWidth, maxHeight);
                for (int k = 0; k < processors.length; k++) {
                    kIp3 = processors[k].createProcessor(maxWidth, maxHeight);
                    kIp3.copyBits(processors[k], (maxWidth / 2) - (processors[k].getWidth() / 2), (maxHeight / 2) - (processors[k].getHeight() / 2), FloatBlitter.ADD);
//                    processors[k].setRoi(processors[k].getWidth() / 2 - minWidth / 2, processors[k].getHeight() / 2 - minHeight / 2, minWidth, minHeight);
//                    kIp3 = processors[k].crop();
                    kIp3 = (landscape) ? kIp3.rotateLeft() : kIp3;
                    stack2.addSlice(kIp3);
                }
                output[c] = new ImagePlus(channels[c].getTitle(), stack2);
            } else if (processors.length == 1) {
                kIp3 = (landscape) ? processors[0].rotateLeft() : processors[0];
                output[c] = new ImagePlus(channels[c].getTitle(), kIp3);
            }
            //

            if (c < luts.length) {
                output[c].setLut(luts[c]);
            }
            output[c].setDisplayRange(rangeMin[c], rangeMax[c]);
            //output[c] = new ImagePlus(channels[c].getTitle(), );
        }
        return getHyperStack(imp.getTitle(), output, true, HYPERSTACK_DEFAULT);
    }

    public static double[] getProfile(ImageProcessor ip, Roi roi, int width, int mode) {
        ImagePlus imp = new ImagePlus("", ip.duplicate());
        double[] pLine;
        mode = (mode > 0) ? mode : 1;
        double[] value;
        //ImProcessor.show(">", ip);
        //RJ.l("t>" + roi.getTypeAsString() + " " + roi.getType() + " : " + width);
        roi = ImPlus.getProfileRoi(roi);

        if (width > 1) {// recupere l'image
            imp.setRoi(roi);
            ImageProcessor sImg = new Straightener().straighten(imp, roi, width);
            int sWidth = sImg.getWidth();
            int sHeight = sImg.getHeight();
            pLine = new double[sWidth];
            for (int i = 0; i < sWidth; i++) {
                value = new double[sHeight];
                for (int j = 0; j < sHeight; j++) {
                    value[j] = (double) sImg.getPixelValue(i, j);
                }

                //"default", "mean", "max", "min"
                switch (mode) {
                    case 1://mean
                        pLine[i] = mean(value, 0);
                        break;
                    case 2://max
                        pLine[i] = max(value, 0);
                        break;
                    case 3://min
                        pLine[i] = min(value, 0);
                        break;
                    default:
                        pLine[i] = max(value, 0);
                        break;

                }
            }
        } else {
            imp.setRoi(roi);
            pLine = new ij.gui.ProfilePlot(imp).getProfile();
        }
        imp.flush();
        return pLine;
    }

    public static double mean(double[] values, double defaultValue) {
        double sum = 0;
        double nb = 0;
        for (int i = 0; i < values.length; i++) {
            if (!Double.isNaN(values[i])) {
                sum += values[i];
                nb++;
            }
        }
        return (nb > 0) ? sum / nb : defaultValue;
    }

    public static double max(double[] values, double defaultValue) {
        double output = Double.NEGATIVE_INFINITY;
        for (int i = 0; i < values.length; i++) {
            if (!Double.isNaN(values[i]) && values[i] > output) {
                output = values[i];
            }
        }
        return (output == Double.NEGATIVE_INFINITY) ? defaultValue : output;
    }

    public static double min(double[] values, double defaultValue) {
        double output = Double.POSITIVE_INFINITY;
        for (int i = 0; i < values.length; i++) {
            if (!Double.isNaN(values[i]) && values[i] < output) {
                output = values[i];
            }
        }
        return output == Double.POSITIVE_INFINITY ? defaultValue : output;
    }

    public static ImagePlus getHyperStack(String title, ImagePlus[] imp1, boolean flush, int mode) {
        int nbSlice = 0;
        int nbChannel = 0;//imp1.length;
        int mWidth = 0;
        int mHeight = 0;
        ImagePlus imp2 = null;
        double[][] displayRange = new double[2][imp1.length];
        LUT[] luts = new LUT[imp1.length];
        int type = 0;
        Calibration calibration = null;

        for (int i = 0; i < imp1.length; i++) {//get the max number of not null image
            if (imp1[i] != null) {
                displayRange[0][i] = imp1[i].getDisplayRangeMin();
                displayRange[1][i] = imp1[i].getDisplayRangeMax();
                nbChannel = i + 1;
                if (calibration == null || !calibration.scaled()) {
                    calibration = imp1[i].getCalibration();
                }

            }
        }

        for (int i = 0; i < nbChannel; i++) {
            if (imp1[i] != null) {
                int[] dim = getDimension(imp1[i]);
                nbSlice = Math.max(nbSlice, dim[6]);
                mWidth = Math.max(mWidth, dim[0]);
                mHeight = Math.max(mHeight, dim[1]);
                imp2 = (imp2 != null) ? imp2 : imp1[i];
                luts[i] = getLut(imp1[i]);
                type = (int) Math.max(type, getTypeProcessor(imp1[i]));
            }
        }

        //force the conversion of Colorpocessor to shortProcessor
        if (type == COLOR_PROCESSOR) {
            type = BYTE_PROCESSOR;
        }

        ImageStack outputStack = new ImageStack(mWidth, mHeight);
        ImageStack stack;
        for (int j = 0; j < nbSlice; j++) {
            for (int i = 0; i < nbChannel; i++) {
                stack = (imp1[i] != null) ? imp1[i].getStack() : null;
                if (stack != null && j < stack.getSize()) {
                    outputStack.addSlice(stack.getSliceLabel(j + 1), convertTo(type, stack.getProcessor(j + 1)));
                } else {
                    outputStack.addSlice("", getEmptyProcessor(type, mWidth, mHeight));
                }
            }
        }
        //IJ.log("flush>");

        if (flush) {
            for (int i = 0; i < nbChannel; i++) {
                if (imp1[i] != null) {
                    imp1[i].close();
                    imp1[i].flush();
                }
            }
        }
        CompositeImage output = null;

        if (outputStack.getSize() > 0) {
            ImagePlus imp3 = new ImagePlus(title, outputStack);
//            ImagePlus imp4 = imp3.duplicate();
//            imp4.show();
            //IJ.log("dim> " + nbChannel + " / " + 1 + " / " + nbSlice);

            switch (mode) {
                case HYPERSTACK_SLICE:
                    imp3.setDimensions(nbChannel, nbSlice, 1);
                    break;
                case HYPERSTACK_FRAME:
                default:
                    imp3.setDimensions(nbChannel, 1, nbSlice);
                    break;
            }

            //imp3.setDimensions(nbChannel, 1, nbSlice);
            output = new CompositeImage(imp3, CompositeImage.COLOR);

            for (int i = 0; i < nbChannel; i++) {
                output.setC(i + 1);
                output.setDisplayRange(displayRange[0][i], displayRange[1][i]);
                /*if (luts[i] != null) {
                 output.setChannelLut(luts[i], i + 1);
                 }*/
            }
            output.setLuts(luts);
            output.setC(1);
            if (calibration != null) {
                output.setCalibration(calibration);
            }
        } else {
            //IJ.showError("Stack Empty");
        }

//        if (output != null) {
//            RJ.log(">" + output);
//            output.show();
//        }
        //imp3.setOpenAsHyperStack(true);
        return output;
    }

    public static LUT getLut(ImagePlus imp) {
        if (imp != null) {
            ImageProcessor ip = imp.getProcessor();
            return getLut(ip);
        }
        return null;
    }

    public static LUT getLut(ImageProcessor ip) {
        if (ip != null) {
            ColorModel cm = ip.getColorModel();
            if (cm instanceof IndexColorModel) {
                LUT lut = new LUT((IndexColorModel) cm, ip.getMin(), ip.getMax());
                return lut;
            } else {
                LUT lut2 = LUT.createLutFromColor(Color.WHITE);
                lut2.min = ip.getMin();
                lut2.max = ip.getMax();
                return lut2;
            }
        }
        return null;
    }

    public static int getTypeProcessor(ImagePlus imp) {
        ImageProcessor ip = (imp != null) ? imp.getProcessor() : null;
        return getTypeProcessor(ip);
    }

    public static int getTypeProcessor(ImageProcessor ip) {
        if (ip instanceof ByteProcessor) {
            return BYTE_PROCESSOR;
        } else if (ip instanceof FloatProcessor) {
            return FLOAT_PROCESSOR;
        } else if (ip instanceof ColorProcessor) {
            return COLOR_PROCESSOR;
        } else {
            return SHORT_PROCESSOR;
        }
    }

    public static int[] getDimension(ImagePlus img) {
        if (img != null) {
            int[] dimension = img.getDimensions(); // (width, height, nChannels, nSlices, nFrames)
            dimension = Arrays.copyOf(dimension, 7);
            int nbMax = 0;
            dimension[5] = 0;
            for (int i = 2; i <= 4; i++) {
                nbMax = Math.max(nbMax, dimension[i]);
                if (dimension[i] > 1) {
                    dimension[5]++;
                }
            }

            if (dimension[5] == 3) {
                dimension[6] = dimension[4];//frames
            } else if (dimension[5] == 2) {
                dimension[6] = Math.max(dimension[3], dimension[4]); // slice
            } else if (dimension[2] > 1) {
                dimension[6] = 1; // slice
            } else {
                dimension[6] = nbMax;
            }
            return dimension;
        }
        return new int[6];
    }

    public static ImageProcessor getEmptyProcessor(int type, int width, int height) {
        switch (type) {
            case BYTE_PROCESSOR:
                return new ByteProcessor(width, height);
            case SHORT_PROCESSOR:
                return new ShortProcessor(width, height);
            case FLOAT_PROCESSOR:
                return new FloatProcessor(width, height);
            case COLOR_PROCESSOR:
                return new ColorProcessor(width, height);
            default:
                return new ShortProcessor(width, height);
        }
    }

    public static ImageProcessor convertTo(int type, ImageProcessor ip) {
        if (ip != null) {
            switch (type) {
                case BYTE_PROCESSOR:
                    return (ip instanceof ByteProcessor) ? ip.duplicate() : ip.convertToByteProcessor();
                case SHORT_PROCESSOR:
                    return (ip instanceof ShortProcessor) ? ip.duplicate() : ip.convertToShortProcessor();
                case FLOAT_PROCESSOR:
                    return (ip instanceof FloatProcessor) ? ip.duplicate() : ip.convertToFloatProcessor();
                case COLOR_PROCESSOR:
                    return (ip instanceof ColorProcessor) ? ip.duplicate() : ip.convertToColorProcessor();
                default:
                    return ip.duplicate();
            }
        }
        return null;
    }
}
